unit DuNoGrad;

{ Unit: DuNoGrad/32  1.11                         (C)1996-1998 by DuNo Electronics I/S }
{ ------------------------------------------------------------------------------------ }
{ Created:  May 4 1996                                                                 }
{ Modified: May 15 1998                                                                }
{ Author:   Johnny Norre, DuNo Electronics I/S                                         }
{ Status:   Freeware:  The message "Portions copyright DuNo Electronics I/S" must be   }
{                      included somewhere in the program (e.g. the about box or the    }
{                      help file) or in the documentation (e.g. the manual or the      }
{                      "read me" file). The copyright notice in the source code must   }
{                      not be changed.                                                 }
{ ------------------------------------------------------------------------------------ }
{ DuNoGrad provides panels with a graduated panel color (like the background of many   }
{ "setup" programs), a graduated caption (like Word) and a general purpose routine for }
{ filling a rectangular area of the screen with a graduated color.                     }
{ ------------------------------------------------------------------------------------ }
{ Version   Date    Description                                                        }
{   0.0   04.05.96  DuNoGrad basic implementation: GradientFillRect, TGradientPanel,   }
{                   TGradientCaption.                                                  }
{   1.0   07.08.96  Original implementation                                            }
{   1.1   11.12.96  Fixed a major problem: v1.0 only allowed one gradientcaption per   }
{                   application. Also ensured the component worked correctly with      }
{                   the different border styles of the mainform.                       }
{   1.11  15.05.98  Modified code to work with Delphi 3                                }
{ ------------------------------------------------------------------------------------ }

interface

uses
  SysUtils,
  WinTypes,
  WinProcs,
  Messages,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  Extctrls;

const
  PALETTEPAGE = 'Additional';  { Where to install the components }

type
  TGradientDirection = (gdLeftRight,gdRightLeft,gdTopBottom,gdBottomTop);

type
  TGradientPanel = class(TPanel)
  private
    { Private declarations }
    FColorSteps : integer;
    FGradientDirection : TGradientDirection;
    FGradientSteps : integer;
    procedure SetColorSteps(Value : integer);
    procedure SetGradientDirection(Value : TGradientDirection);
    procedure SetGradientSteps(Value : integer);
  protected
    { Protected declarations }
    constructor Create(AOwner : TComponent); override;
    procedure Paint; override;
  public
    { Public declarations }
  published
    { Published declarations }
    property ColorSteps : integer read FColorSteps write SetColorSteps default 256;
    property GradientDirection : TGradientDirection read FGradientDirection write SetGradientDirection default gdTopBottom;
    property GradientSteps : integer read FGradientSteps write SetGradientSteps default 256;
  end;
  { The following class is used to get access to the DefWndProc property in the owner form }
  { WARNING: This method can not be considered "correct" programming - it stretches the    }
  {          rules of Object Pascal to the limit - it might not even work in future        }
  {          versions of Delphi!                                                           }
  _dunox_TWinControl = class(TWinControl)
  public
    property DefWndProc;
  end;
  TGradientCaption = class(TComponent)
  private
    { Private declarations }
    OrgWndProc : pointer;
    OwnerForm : TForm;
    FCheckWindows98 : boolean;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor  Destroy; override;
    procedure   Loaded; override;
  published
    { Published declarations }
    property CheckWindows98 : boolean read FCheckWindows98 write FCheckWindows98; { Only works at design time! }
  end;

procedure GradientFillRect(Canvas : TCanvas; const Rect : TRect; Direction : TGradientDirection;
                           Steps : integer; Color : TColor; ColorSteps : integer);

procedure Register;

implementation

{ Used by the AddHandle/RemoveHandle/FindHandle functions to keep track }
{ of the forms using a graduated caption.                               }
type
  PCaptionRec = ^TCaptionRec;
  TCaptionRec = record
    Caption : TGradientCaption;
    Next : PCaptionRec;
  end;

var
  CaptionList : PCaptionRec;

{ Add a handle to the handle list }  
function AddHandle(Caption : TGradientCaption) : boolean;
var
  P : PCaptionRec;
begin
  AddHandle := false;
  try
    new(P);
    P^.Caption := Caption;
    P^.Next := CaptionList;
    CaptionList := P;
    AddHandle := true
  except
    on EOutOfMemory do ;
  end;
end;

{ Remove a handle from the handle list }
function RemoveHandle(Caption : TGradientCaption) : boolean;
var
  PointToP,
  P : PCaptionRec;
begin
  RemoveHandle := false;
  PointToP := nil;
  P := CaptionList;
  while (P <> nil) and (P^.Caption <> Caption) do begin
    PointToP := P;
    P := P^.Next
  end;
  if P = nil then
    exit;
  if PointToP = nil then
    CaptionList := P^.Next
  else
    PointToP^.Next := P^.Next;
  dispose(P)
end;

{ Find a handle from a HWnd in the handle list }
function FindHandle(Wnd : HWnd) : TGradientCaption;
var
  P : PCaptionRec;
begin
  FindHandle := nil;
  P := CaptionList;
  while (P <> nil) and (P^.Caption.OwnerForm.Handle <> Wnd) do
    P := P^.Next;
  if P = nil then
    exit;
  FindHandle := P^.Caption
end;

{ Check if this is Windows 98 }
function IsWin98 : boolean;
var
  OSVer : TOSVersionInfo;
begin
  OSVer.dwOSVersionInfoSize := sizeof(TOSVersionInfo);
  GetVersionEx(OSVer);
  { Check the platform id is "Windows 95" and that the version >= 4.10 }
  IsWin98 := (OSVer.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS) and (OSVer.dwMajorVersion >= 4) and (OSVer.dwMinorVersion >= 10)	
end;

{ Fill a rectangle with a graduated color        }
{ Canvas     is the canvas to paint on           }
{ Rect       the coordinates of the area to fill }
{ Direction  the direction of the gradient       }
{ Steps      the number of color bands           }
{ Color      the starting color                  }
{ ColorSteps the number of color steps - if      }
{            ColorSteps < Steps, the color will  }
{            not fade all the way to black       }
procedure GradientFillRect(Canvas : TCanvas; const Rect : TRect; Direction : TGradientDirection;
                           Steps : integer; Color : TColor; ColorSteps : integer);
var
  I : integer;
  dStep : double;
  RStep,
  GStep,
  BStep : double;
  RStart,
  GStart,
  BStart : integer;
  FRect : TRect;
begin
  { Make sure the supplied values are within sensible limits }
  if (ColorSteps < 0) or (ColorSteps > Steps) then
    ColorSteps := Steps;  { Because we can't graduate to anything darker than black }
  if Steps = 0 then
    Steps := 1;
  if ColorSteps = 0 then
    ColorSteps := 1;
  { Calculate the number of pixels per color band }
  if Direction in [GDLeftRight,GDRightLeft] then
    dStep := (Rect.Right-Rect.Left) / Steps
  else
    dStep := (Rect.Bottom-Rect.Top) / Steps;
  Color := ColorToRGB(Color);
  { We fade each color component separately - calculate the }
  { steps and color difference for each color }
  RStep := (Color mod 256) / Steps;
  GStep := ((Color and $FF00) shr 8) / Steps;
  BStep := (Color div 65536) / Steps ;
  RStart := trunc(RStep*(Steps-ColorSteps));
  GStart := trunc(GStep*(Steps-ColorSteps));
  BStart := trunc(BStep*(Steps-ColorSteps));
  RStep := RStep / (Steps/ColorSteps);
  GStep := GStep / (Steps/ColorSteps);
  BStep := BStep / (Steps/ColorSteps);
  FRect := Rect;
  { Colorize the bands }
  for I := 0 to Steps do begin
    case Direction of
      GDLeftRight :
        FRect.Right := Rect.Left+trunc(dStep*I);
      GDRightLeft :
        FRect.Left := Rect.Right-Rect.Left-trunc(dStep*I);
      GDTopBottom :
        FRect.Bottom := Rect.Top+trunc(dStep*I);
      GDBottomTop :
        FRect.Top := Rect.Bottom-Rect.Top-trunc(dStep*I);
    end;
    if FRect.Left < Rect.Left then
      FRect.Left := Rect.Left;
    if FRect.Right > Rect.Right then
      FRect.Right := Rect.Right;
    if FRect.Top < Rect.Top then
      FRect.TOp := Rect.Top;
    if FRect.Bottom > Rect.Bottom then
      FRect.Bottom := Rect.Bottom;
    Canvas.Brush.Color := Rgb(trunc((Steps-I)*RStep)+RStart,
                              trunc((Steps-I)*GStep)+GStart,
                              trunc((Steps-I)*BStep)+BStart);
    Canvas.FillRect(FRect);
    case Direction of
      GDLeftRight :
        FRect.Left := FRect.Right;
      GDRightLeft :
        FRect.Right := FRect.Left;
      GDTopBottom :
        FRect.Top := FRect.Bottom;
      GDBottomTop :
        FRect.Bottom := FRect.Top;
    end;
  end
end;


{ ********  TGRADIENTPANEL  ******** }

constructor TGradientPanel.Create;
begin
  inherited Create(AOwner);
  FColorSteps := 256;
  FGradientDirection := gdTopBottom;
  FGradientSteps := 256;
end;

procedure TGradientPanel.Paint;
const
  Alignments : array[TAlignment] of word = (DT_LEFT, DT_RIGHT, DT_CENTER); { Convert Delphi TAlignment to Windows constants }
var
  Rect: TRect; { The rectangle to fill }
  TopColor,
  BottomColor: TColor; { Used for the 3D effects of the bevel }
  FontHeight: Integer; { The height in pixels of the font selected }
  Text : string;  { The text to draw in the panel }

  { Adjust the colors for the bevel }
  procedure AdjustColors(Bevel: TPanelBevel);
  begin
    TopColor := clBtnHighlight;
    if Bevel = bvLowered then
      TopColor := clBtnShadow;
    BottomColor := clBtnShadow;
    if Bevel = bvLowered then
      BottomColor := clBtnHighlight;
  end;

begin
  { Paint the bevel of the panel (if any) }
  Rect := GetClientRect;
  if BevelOuter <> bvNone then begin
    AdjustColors(BevelOuter);
    Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
  end;
  Frame3D(Canvas, Rect, Color, Color, BorderWidth);
  if BevelInner <> bvNone then begin
    AdjustColors(BevelInner);
    Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
  end;
  { Fill the client area and draw the text }
  with Canvas do begin
    GradientFillRect(Canvas,Rect,FGradientDirection,FGradientSteps,Color,FColorSteps);
    Brush.Color := Color;
    Brush.Style := bsClear;
    Font := Self.Font;
    FontHeight := TextHeight('W');
    with Rect do begin
      Top := ((Bottom + Top) - FontHeight) div 2;
      Bottom := Top + FontHeight;
    end;
    Text := Caption+#0;
    DrawText(Handle, @Text[1], -1, Rect, (DT_EXPANDTABS or
      DT_VCENTER) or Alignments[Alignment]);
  end;
end;

procedure TGradientPanel.SetColorSteps(Value : integer);
begin
  if (Value > 0) and (Value <= 256) then begin
    if (Value <> FColorSteps) then begin
      FColorSteps := Value;
      Invalidate
    end
  end else
    Application.MessageBox('Illegal value (1..256)','TGradientPanel.ColorSteps',0)
end;

procedure TGradientPanel.SetGradientDirection(Value : TGradientDirection);
begin
  if (Value <> FGradientDirection) then begin
    FGradientDirection := Value;
    Invalidate
  end
end;

procedure TGradientPanel.SetGradientSteps(Value : integer);
begin
  if (Value > 0) and (Value <= 256) then begin
    if (Value <> FGradientSteps) then begin
      FGradientSteps := Value;
      Invalidate
    end
  end else
    Application.MessageBox('Illegal value (1..256)','TGradientPanel.GradientSteps',0)
end;

{ NewWndProc is used by TGradientCaption to paint the caption of the windows }
function NewWndProc(Wnd : HWnd; Msg, wParam : word; lParam : longint) : longint; stdcall; far;
var
  Caption : TGradientCaption;

  { DrawWinCaption draw the actual graduated caption }
  procedure DrawWinCaption;
  var
    Canvas : TCanvas;
    DC : HDC;
    WR,
    Rect : TRect;
    LeftIcons,
    RightIcons : integer;
    S : array[0..255] of char;
    Ncm : TNonClientMetrics;
  begin
    if Caption.OwnerForm.BorderStyle = bsNone then
      exit;
    Canvas := TCanvas.Create;
    NewWndProc := CallWindowProc(Caption.OrgWndProc,Wnd,Msg,wParam,lParam);
    with Caption.OwnerForm  do begin
      if biSystemMenu in BorderIcons then
        LeftIcons := 1
      else
        LeftIcons := 0;
      if biMinimize in BorderIcons then
        RightIcons := 1
      else
        RightIcons := 0;
      if biMaximize in BorderIcons then
        inc(RightIcons);
      if (RightIcons <> 0) then
        RightIcons := 2;
      inc(RightIcons);
      if biHelp in BorderIcons then
        if RightIcons < 3 then
          inc(RightIcons);
    end;
    DC := GetWindowDC(Wnd);
    Canvas.Handle := DC;
    GetWindowRect(Wnd,WR);
    if (Caption.OwnerForm.BorderStyle = bsSizeable) or (Caption.OwnerForm.BorderStyle = bsSizeToolWin) then
      Rect.Top := GetSystemMetrics(SM_CYFRAME){!}{!1!} { Forward }
    else if (Caption.OwnerForm.BorderStyle = bsDialog) then
      Rect.Top := GetSystemMetrics(SM_CYDLGFRAME)
    else
      Rect.Top := GetSystemMetrics(SM_CYBORDER)+2;
    Rect.Left := GetSystemMetrics(SM_CXBORDER)+GetSystemMetrics(SM_CXFRAME)-1;
    if Caption.OwnerForm.BorderStyle = bsDialog then
      Rect.Bottom := Rect.Top+GetSystemMetrics(SM_CYCAPTION)-1
    else if (Caption.OwnerForm.BorderStyle = bsToolWindow) or (Caption.OwnerForm.BorderStyle = bsSizeToolWin) then
      Rect.Bottom := Rect.Top+GetSystemMetrics(SM_CYSMCAPTION)
    else
      Rect.Bottom := Rect.Top+GetSystemMetrics(SM_CYCAPTION);
    Rect.Right := WR.Right - WR.Left-(GetSystemMetrics(SM_CXFRAME)+
                  RightIcons*GetSystemMetrics(SM_CXSIZE));
    GradientFillRect(Canvas,Rect,gdRightLeft,256,GetSysColor(COLOR_ACTIVECAPTION),256);
    if (LeftIcons > 0) then
      PostMessage(Wnd,WM_SETICON,0,0);
    Rect.Left := GetSystemMetrics(SM_CXBORDER)+GetSystemMetrics(SM_CXFRAME)+LeftIcons*GetSystemMetrics(SM_CXSIZE)-1;
    SetTextColor(Canvas.Handle,GetSysColor(COLOR_CAPTIONTEXT));
    Ncm.cbSize := sizeof(TNonClientMetrics);
    if SystemParametersInfo(SPI_GETNONCLIENTMETRICS,sizeof(TNonClientMetrics),@Ncm,0) then begin
      Canvas.Font.Name := strpas(Ncm.lfCaptionFont.lfFaceName);
      Canvas.Font.Height := Ncm.lfCaptionFont.lfHeight;
      if Ncm.lfCaptionFont.lfWeight > 400 then
        Canvas.Font.Style := [fsBold]
      else
        Canvas.Font.Style := [];
      if Ncm.lfCaptionFont.lfItalic <> 0 then
        Canvas.Font.Style := Canvas.Font.Style + [fsItalic];
    end else begin
      Canvas.Font.Name := 'System';
      Canvas.Font.Style := [fsBold];
      Canvas.Font.Size := 10;
    end;
    Canvas.Font.Color := clWhite{GetSysColor(COLOR_CAPTIONTEXT)};
    GetWindowText(Wnd,S,255);
    SetBkMode(Canvas.Handle,TRANSPARENT);
    if ((Caption.OwnerForm.BorderStyle = bsToolWindow) or (Caption.OwnerForm.BorderStyle = bsSizeToolWin) or (Caption.OwnerForm.BorderStyle = bsDialog)) and (LeftIcons > 0) then
      Rect.Left := Rect.Left - GetSystemMetrics(SM_CXSIZE);
    Canvas.TextOut(Rect.Left+2,Rect.Top+2,strpas(S));
    Canvas.Handle := 0;
    ReleaseDC(Wnd,DC);
    Canvas.Free
  end;
begin
  { Locate the form }
  Caption := FindHandle(Wnd);
  { If no form was found, call DefWindowProc - this should never be called }
  { while the program is running - but might be called during start up and }
  { shut down.                                                             }
  if Caption = nil then begin
    NewWndProc := DefWindowProc(Wnd,Msg,wParam,lParam);
    exit
  end;
  { Capture WM_NCACTIVATE (Non-client area of form (=caption) is activated  }
  { or deactivated) to change between a colored and a gray caption. Capture }
  { WM_NCPAINT messages to paint and repaint the caption.                   }
  case Msg of
    WM_NCACTIVATE : begin
      if wParam <> 0 then begin
        DrawWinCaption;
        NewWndProc := 0;
        exit
      end;
    end;
    WM_NCPAINT : begin
      if Caption.OwnerForm.Active then begin
        DrawWinCaption;
        NewWndProc := 0;
        exit
      end
    end;
  end;
  { If message wasn't a WM_NCPAINT or a WM_NCACTIVATE, then call the old window procedure }
  NewWndProc := CallWindowProc(Caption.OrgWndProc,Wnd,Msg,wParam,lParam);
end;


{ ********  TGRADIENTCAPTION  ******** }

constructor TGradientCaption.Create;
begin
  inherited Create(AOwner);
  if not (csDesigning in ComponentState) and (AOwner is TForm) then
    OwnerForm := TForm(AOwner);
end;

destructor TGradientCaption.Destroy;
begin
  inherited Destroy;
  if not (csDesigning in ComponentState) and (Owner is TForm) then begin
    _dunox_TWinControl(OwnerForm).DefWndProc := OrgWndProc;
    RemoveHandle(self);
  end
end;

procedure TGradientCaption.Loaded;
begin
  if not (csDesigning in ComponentState) and (Owner is TForm) then begin
    if FCheckWindows98 and IsWin98 then
      exit;
    if AddHandle(self) then begin
      OrgWndProc := _dunox_TWinControl(Owner).DefWndProc;
      _dunox_TWinControl(Owner).DefWndProc := @NewWndProc;
    end
  end
end;

procedure Register;
begin
  RegisterComponents(PALETTEPAGE, [TGradientPanel,TGradientCaption]);
end;

begin
  CaptionList := nil
end.
